import React, { useState } from 'react';
import { Link } from 'react-router-dom';
import { motion } from 'framer-motion';
import { 
  Scale, 
  ArrowLeft, 
  Edit3, 
  Sparkles,
  Copy,
  Save,
  RefreshCw
} from 'lucide-react';
import { Button } from '@/components/ui/button.jsx';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card.jsx';
import { Input } from '@/components/ui/input.jsx';
import { Label } from '@/components/ui/label.jsx';
import { Textarea } from '@/components/ui/textarea.jsx';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select.jsx';
import { Badge } from '@/components/ui/badge.jsx';

const clauseTypes = [
  'Confidentiality Clause',
  'Termination Clause',
  'Payment Terms',
  'Liability Limitation',
  'Force Majeure',
  'Dispute Resolution',
  'Intellectual Property',
  'Non-Compete',
  'Governing Law',
  'Amendment Clause',
  'Indemnification',
  'Warranty Disclaimer'
];

const jurisdictions = [
  { value: 'SA', label: 'Saudi Arabia' },
  { value: 'JO', label: 'Jordan' }
];

function ClauseDrafter() {
  const [formData, setFormData] = useState({
    clauseType: '',
    jurisdiction: '',
    context: '',
    specificRequirements: ''
  });
  
  const [draftedClause, setDraftedClause] = useState('');
  const [isDrafting, setIsDrafting] = useState(false);
  const [suggestions, setSuggestions] = useState([]);

  const handleInputChange = (field, value) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const draftClause = async () => {
    setIsDrafting(true);
    
    // Simulate API call
    setTimeout(() => {
      const mockClause = generateMockClause(formData.clauseType, formData.jurisdiction);
      setDraftedClause(mockClause);
      
      // Generate suggestions
      const mockSuggestions = [
        'Consider adding specific timeframes for notice periods',
        'Include provisions for electronic communications',
        'Add force majeure exceptions if applicable',
        'Specify governing law jurisdiction clearly'
      ];
      setSuggestions(mockSuggestions);
      
      setIsDrafting(false);
    }, 2000);
  };

  const generateMockClause = (clauseType, jurisdiction) => {
    const jurisdictionName = jurisdiction === 'SA' ? 'Saudi Arabia' : 'Jordan';
    
    switch (clauseType) {
      case 'Confidentiality Clause':
        return `CONFIDENTIALITY

Each party acknowledges that it may have access to certain confidential information of the other party. Each party agrees to maintain in confidence any confidential information received from the other party and not to disclose such confidential information to third parties without the prior written consent of the disclosing party.

For the purposes of this Agreement, "Confidential Information" shall include all non-public, proprietary or confidential information, technical data, trade secrets, know-how, research, product plans, products, services, customers, customer lists, markets, software, developments, inventions, processes, formulas, technology, designs, drawings, engineering, hardware configuration information, marketing, finances, or other business information.

This obligation shall survive termination of this Agreement and shall continue for a period of [5] years from the date of disclosure, unless otherwise required by the laws of ${jurisdictionName}.

The receiving party shall use the same degree of care to protect the confidential information as it uses to protect its own confidential information, but in no event less than reasonable care.`;

      case 'Termination Clause':
        return `TERMINATION

This Agreement may be terminated:

(a) By mutual written consent of both parties;

(b) By either party upon [30] days' written notice to the other party;

(c) By either party immediately upon written notice if the other party materially breaches this Agreement and fails to cure such breach within [15] days after receiving written notice of the breach;

(d) By either party immediately upon written notice if the other party becomes insolvent, makes an assignment for the benefit of creditors, files for bankruptcy, or has a receiver appointed for its business or assets.

Upon termination of this Agreement, each party shall return or destroy all confidential information of the other party in its possession or control.

The provisions of this Agreement that by their nature should survive termination shall survive, including but not limited to confidentiality, limitation of liability, and governing law provisions.

This termination clause is governed by the laws of ${jurisdictionName}.`;

      case 'Payment Terms':
        return `PAYMENT TERMS

Payment for services rendered under this Agreement shall be made according to the following terms:

(a) All payments shall be made in [Currency] within [30] days of receipt of invoice;

(b) Invoices shall be submitted monthly and shall include detailed descriptions of services performed;

(c) Late payments shall incur interest at the rate of [1.5%] per month or the maximum rate permitted by law in ${jurisdictionName}, whichever is lower;

(d) All payments shall be made by bank transfer to the account specified by the payee;

(e) Each party shall be responsible for its own banking fees and charges;

(f) In case of disputed invoices, the undisputed portion shall be paid within the specified timeframe, and the disputed portion shall be resolved through good faith negotiations.

All payment obligations are subject to applicable tax laws and regulations of ${jurisdictionName}.`;

      case 'Dispute Resolution':
        return `DISPUTE RESOLUTION

Any dispute, controversy, or claim arising out of or relating to this Agreement, or the breach, termination, or invalidity thereof, shall be resolved as follows:

(a) NEGOTIATION: The parties shall first attempt to resolve any dispute through good faith negotiations for a period of [30] days;

(b) MEDIATION: If negotiation fails, the parties agree to submit the dispute to mediation before a mutually agreed mediator in ${jurisdictionName};

(c) ARBITRATION: If mediation is unsuccessful, any remaining dispute shall be finally settled by arbitration under the rules of [applicable arbitration institution] in ${jurisdictionName};

(d) The arbitration shall be conducted in [Arabic/English] language;

(e) The arbitral award shall be final and binding upon both parties;

(f) Each party shall bear its own costs and expenses, and the parties shall share equally the fees and expenses of the arbitrator(s);

(g) This dispute resolution clause shall be governed by the laws of ${jurisdictionName}.

Nothing in this clause shall prevent either party from seeking interim or injunctive relief in a court of competent jurisdiction.`;

      default:
        return `[${clauseType.toUpperCase()}]

This clause shall be drafted in accordance with the laws and regulations of ${jurisdictionName}. The specific terms and conditions shall be tailored to meet the requirements of this Agreement and ensure compliance with applicable legal standards.

[Detailed clause content would be generated here based on the specific requirements and context provided.]

This clause is subject to the governing law of ${jurisdictionName} and shall be interpreted in accordance with local legal principles and practices.`;
    }
  };

  const copyToClipboard = () => {
    navigator.clipboard.writeText(draftedClause);
    alert('Clause copied to clipboard!');
  };

  const saveClause = () => {
    // Simulate saving clause
    alert('Clause saved successfully!');
  };

  const regenerateClause = () => {
    if (formData.clauseType && formData.jurisdiction) {
      draftClause();
    }
  };

  return (
    <div className="flex">
      {/* Sidebar */}
      <div className="w-64 bg-white shadow-lg h-screen fixed left-0 top-0 z-10">
        <div className="p-6 border-b">
          <Link to="/" className="flex items-center space-x-2 text-gray-600 hover:text-gray-900">
            <ArrowLeft className="h-5 w-5" />
            <span>Back to Dashboard</span>
          </Link>
        </div>
        <div className="p-6">
          <div className="flex items-center space-x-2 mb-4">
            <Scale className="h-6 w-6 text-blue-600" />
            <span className="text-xl font-bold text-gray-900">LAWAI</span>
          </div>
          <Badge variant="secondary" className="w-full justify-center">
            Clause Drafter
          </Badge>
        </div>
      </div>

      {/* Main Content */}
      <div className="flex-1 ml-64 p-8">
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5 }}
        >
          <div className="mb-8">
            <h1 className="text-3xl font-bold text-gray-900 mb-2">Draft Clauses</h1>
            <p className="text-gray-600">Create specific contract clauses with AI assistance</p>
          </div>

          <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
            {/* Input Section */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center space-x-2">
                  <Edit3 className="h-5 w-5" />
                  <span>Clause Requirements</span>
                </CardTitle>
                <CardDescription>
                  Specify the type and requirements for your clause
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                <div>
                  <Label htmlFor="clauseType">Clause Type</Label>
                  <Select onValueChange={(value) => handleInputChange('clauseType', value)}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select clause type" />
                    </SelectTrigger>
                    <SelectContent>
                      {clauseTypes.map((type) => (
                        <SelectItem key={type} value={type}>
                          {type}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <Label htmlFor="jurisdiction">Jurisdiction</Label>
                  <Select onValueChange={(value) => handleInputChange('jurisdiction', value)}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select jurisdiction" />
                    </SelectTrigger>
                    <SelectContent>
                      {jurisdictions.map((jurisdiction) => (
                        <SelectItem key={jurisdiction.value} value={jurisdiction.value}>
                          {jurisdiction.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <Label htmlFor="context">Contract Context</Label>
                  <Textarea
                    id="context"
                    placeholder="Describe the overall contract context (e.g., employment agreement, service contract, etc.)"
                    rows={4}
                    value={formData.context}
                    onChange={(e) => handleInputChange('context', e.target.value)}
                  />
                </div>

                <div>
                  <Label htmlFor="specificRequirements">Specific Requirements</Label>
                  <Textarea
                    id="specificRequirements"
                    placeholder="Describe any specific requirements, conditions, or customizations needed for this clause..."
                    rows={4}
                    value={formData.specificRequirements}
                    onChange={(e) => handleInputChange('specificRequirements', e.target.value)}
                  />
                </div>

                <Button 
                  onClick={draftClause}
                  disabled={!formData.clauseType || !formData.jurisdiction || isDrafting}
                  className="w-full"
                >
                  {isDrafting ? (
                    <>
                      <Sparkles className="h-4 w-4 mr-2 animate-spin" />
                      Drafting Clause...
                    </>
                  ) : (
                    <>
                      <Sparkles className="h-4 w-4 mr-2" />
                      Draft Clause
                    </>
                  )}
                </Button>
              </CardContent>
            </Card>

            {/* Generated Clause Section */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center space-x-2">
                  <Edit3 className="h-5 w-5" />
                  <span>Generated Clause</span>
                </CardTitle>
                <CardDescription>
                  Your AI-drafted clause will appear here
                </CardDescription>
              </CardHeader>
              <CardContent>
                {draftedClause ? (
                  <div className="space-y-4">
                    <div className="flex space-x-2">
                      <Button onClick={copyToClipboard} variant="outline" size="sm">
                        <Copy className="h-4 w-4 mr-2" />
                        Copy
                      </Button>
                      <Button onClick={saveClause} variant="outline" size="sm">
                        <Save className="h-4 w-4 mr-2" />
                        Save
                      </Button>
                      <Button onClick={regenerateClause} variant="outline" size="sm">
                        <RefreshCw className="h-4 w-4 mr-2" />
                        Regenerate
                      </Button>
                    </div>
                    
                    <div className="bg-gray-50 p-4 rounded-lg max-h-96 overflow-y-auto">
                      <pre className="whitespace-pre-wrap text-sm text-gray-800">
                        {draftedClause}
                      </pre>
                    </div>

                    {suggestions.length > 0 && (
                      <div className="mt-6">
                        <h3 className="font-semibold mb-3 text-blue-600">AI Suggestions</h3>
                        <ul className="space-y-2">
                          {suggestions.map((suggestion, index) => (
                            <li key={index} className="text-sm text-gray-600 bg-blue-50 p-3 rounded-lg">
                              💡 {suggestion}
                            </li>
                          ))}
                        </ul>
                      </div>
                    )}
                  </div>
                ) : (
                  <div className="text-center py-12 text-gray-500">
                    <Edit3 className="h-12 w-12 mx-auto mb-4 opacity-50" />
                    <p>Select clause type and jurisdiction, then click "Draft Clause" to generate your clause</p>
                  </div>
                )}
              </CardContent>
            </Card>
          </div>

          {/* Quick Clause Templates */}
          <Card className="mt-8">
            <CardHeader>
              <CardTitle>Quick Clause Templates</CardTitle>
              <CardDescription>
                Common clause types for quick access
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                {clauseTypes.slice(0, 8).map((type) => (
                  <Button
                    key={type}
                    variant="outline"
                    size="sm"
                    onClick={() => {
                      handleInputChange('clauseType', type);
                      if (formData.jurisdiction) {
                        setTimeout(() => draftClause(), 100);
                      }
                    }}
                    className="h-auto p-3 text-left justify-start"
                  >
                    <div>
                      <div className="font-medium text-sm">{type}</div>
                    </div>
                  </Button>
                ))}
              </div>
            </CardContent>
          </Card>
        </motion.div>
      </div>
    </div>
  );
}

export default ClauseDrafter;

