import { FileText, ArrowLeft, Scale, Download } from 'lucide-react';
import { saveAs } from 'file-saver';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card.jsx';
import { Button } from '@/components/ui/button.jsx';
import { API_BASE_URL } from "../config";

function ContractDetail() {
  const { id } = useParams();
  const [contract, setContract] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  useEffect(() => {
    const fetchContract = async () => {
      try {
        const response = await fetch(`${API_BASE_URL}/api/contracts/${id}`);
        if (!response.ok) {
          throw new Error(`HTTP error! status: ${response.status}`);
        }
        const data = await response.json();
        setContract(data);
      } catch (error) {
        setError(error);
        console.error("Error fetching contract details:", error);
      } finally {
        setLoading(false);
      }
    };

    fetchContract();
  }, [id]);

  const handleDownloadWord = async () => {
    try {
      const response = await fetch(`${API_BASE_URL}/api/contracts/${id}/download_word`);
      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }
      const blob = await response.blob();
      saveAs(blob, `${contract.title}.docx`);
    } catch (error) {
      console.error("Error downloading Word document:", error);
      alert("Failed to download Word document.");
    }
  };

  if (loading) {
    return <div className="flex-1 ml-64 p-8 text-center">Loading contract details...</div>;
  }

  if (error) {
    return <div className="flex-1 ml-64 p-8 text-center text-red-500">Error: {error.message}</div>;
  }

  if (!contract) {
    return <div className="flex-1 ml-64 p-8 text-center">Contract not found.</div>;
  }

  return (
    <div className="flex">
      {/* Sidebar */}
      <div className="w-64 bg-white shadow-lg h-screen fixed left-0 top-0 z-10">
        <div className="p-6 border-b">
          <Link to="/" className="flex items-center space-x-2 text-gray-600 hover:text-gray-900">
            <ArrowLeft className="h-5 w-5" />
            <span>Back to Dashboard</span>
          </Link>
        </div>
        <div className="p-6">
          <div className="flex items-center space-x-2 mb-4">
            <Scale className="h-6 w-6 text-blue-600" />
            <span className="text-xl font-bold text-gray-900">LAWAI</span>
          </div>
        </div>
      </div>

      {/* Main Content */}
      <div className="flex-1 ml-64 p-8">
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5 }}
        >
          <div className="mb-8">
            <h1 className="text-3xl font-bold text-gray-900 mb-2">Contract Details</h1>
            <p className="text-gray-600">View the full content and details of your contract.</p>
          </div>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center space-x-2">
                <FileText className="h-5 w-5" />
                <span>{contract.title}</span>
              </CardTitle>
              <CardDescription>
                Type: {contract.contract_type} • Jurisdiction: {contract.jurisdiction === 'SA' ? 'Saudi Arabia' : 'Jordan'}
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div>
                <h3 className="text-lg font-semibold">Content:</h3>
                <div className="bg-gray-50 p-4 rounded-lg max-h-96 overflow-y-auto">
                  <pre className="whitespace-pre-wrap text-sm text-gray-800">
                    {contract.content}
                  </pre>
                </div>
              </div>
              <div>
                <h3 className="text-lg font-semibold">Created At:</h3>
                <p>{new Date(contract.created_at).toLocaleString()}</p>
              </div>
              <div>
                <Button onClick={handleDownloadWord} className="mt-4"><Download className="h-4 w-4 mr-2" />Download as Word</Button>
              </div>
              {/* Add more details as needed */}
            </CardContent>
          </Card>
        </motion.div>
      </div>
    </div>
  );
}

export default ContractDetail;
