import React, { useState } from 'react';
import { Link } from 'react-router-dom';
import { motion } from 'framer-motion';
import { 
  Scale, 
  ArrowLeft, 
  Search, 
  Upload,
  AlertTriangle,
  CheckCircle,
  XCircle,
  FileText,
  Download
} from 'lucide-react';
import { Button } from '@/components/ui/button.jsx';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card.jsx';
import { Textarea } from '@/components/ui/textarea.jsx';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select.jsx';
import { Badge } from '@/components/ui/badge.jsx';
import { Alert, AlertDescription } from '@/components/ui/alert.jsx';

const jurisdictions = [
  { value: 'SA', label: 'Saudi Arabia' },
  { value: 'JO', label: 'Jordan' }
];

const reviewTypes = [
  { value: 'general', label: 'General Review' },
  { value: 'compliance', label: 'Compliance Check' },
  { value: 'risk', label: 'Risk Assessment' }
];

function ContractReview() {
  const [contractText, setContractText] = useState('');
  const [jurisdiction, setJurisdiction] = useState('');
  const [reviewType, setReviewType] = useState('general');
  const [reviewResults, setReviewResults] = useState(null);
  const [isReviewing, setIsReviewing] = useState(false);

  const handleFileUpload = (event) => {
    const file = event.target.files[0];
    if (file) {
      const reader = new FileReader();
      reader.onload = (e) => {
        setContractText(e.target.result);
      };
      reader.readAsText(file);
    }
  };

  const reviewContract = async () => {
    setIsReviewing(true);
    
    // Simulate API call
    setTimeout(() => {
      const mockResults = {
        risk_level: 'Medium',
        summary: 'The contract has been reviewed and several areas require attention. While the overall structure is sound, there are compliance issues and potential risks that should be addressed.',
        compliance_issues: [
          'Missing force majeure clause required under Saudi Arabian commercial law',
          'Dispute resolution mechanism does not specify arbitration location',
          'Payment terms may conflict with local banking regulations'
        ],
        legal_risks: [
          'Unlimited liability exposure for both parties',
          'Vague termination conditions could lead to disputes',
          'Intellectual property ownership not clearly defined'
        ],
        missing_clauses: [
          'Confidentiality and non-disclosure provisions',
          'Data protection and privacy compliance',
          'Governing law and jurisdiction clause',
          'Amendment and modification procedures'
        ],
        ambiguous_language: [
          'Section 3.2: "reasonable efforts" needs specific definition',
          'Section 5.1: "timely manner" should specify exact timeframes',
          'Section 7.3: "material breach" requires clear criteria'
        ],
        recommendations: [
          'Add comprehensive force majeure clause compliant with local law',
          'Define all ambiguous terms in a definitions section',
          'Include specific dispute resolution procedures',
          'Add data protection compliance clauses',
          'Specify liability caps and limitations',
          'Include detailed termination procedures'
        ]
      };
      
      setReviewResults(mockResults);
      setIsReviewing(false);
    }, 3000);
  };

  const getRiskColor = (riskLevel) => {
    switch (riskLevel?.toLowerCase()) {
      case 'low': return 'text-green-600 bg-green-100';
      case 'medium': return 'text-yellow-600 bg-yellow-100';
      case 'high': return 'text-red-600 bg-red-100';
      default: return 'text-gray-600 bg-gray-100';
    }
  };

  const getRiskIcon = (riskLevel) => {
    switch (riskLevel?.toLowerCase()) {
      case 'low': return CheckCircle;
      case 'medium': return AlertTriangle;
      case 'high': return XCircle;
      default: return AlertTriangle;
    }
  };

  const downloadReport = () => {
    if (!reviewResults) return;
    
    const report = `
CONTRACT REVIEW REPORT
Generated by LAWAI

OVERALL RISK LEVEL: ${reviewResults.risk_level}

SUMMARY:
${reviewResults.summary}

COMPLIANCE ISSUES:
${reviewResults.compliance_issues.map((issue, index) => `${index + 1}. ${issue}`).join('\n')}

LEGAL RISKS:
${reviewResults.legal_risks.map((risk, index) => `${index + 1}. ${risk}`).join('\n')}

MISSING CLAUSES:
${reviewResults.missing_clauses.map((clause, index) => `${index + 1}. ${clause}`).join('\n')}

AMBIGUOUS LANGUAGE:
${reviewResults.ambiguous_language.map((item, index) => `${index + 1}. ${item}`).join('\n')}

RECOMMENDATIONS:
${reviewResults.recommendations.map((rec, index) => `${index + 1}. ${rec}`).join('\n')}

Report generated on: ${new Date().toLocaleDateString()}
    `;
    
    const element = document.createElement('a');
    const file = new Blob([report], { type: 'text/plain' });
    element.href = URL.createObjectURL(file);
    element.download = 'contract_review_report.txt';
    document.body.appendChild(element);
    element.click();
    document.body.removeChild(element);
  };

  return (
    <div className="flex">
      {/* Sidebar */}
      <div className="w-64 bg-white shadow-lg h-screen fixed left-0 top-0 z-10">
        <div className="p-6 border-b">
          <Link to="/" className="flex items-center space-x-2 text-gray-600 hover:text-gray-900">
            <ArrowLeft className="h-5 w-5" />
            <span>Back to Dashboard</span>
          </Link>
        </div>
        <div className="p-6">
          <div className="flex items-center space-x-2 mb-4">
            <Scale className="h-6 w-6 text-blue-600" />
            <span className="text-xl font-bold text-gray-900">LAWAI</span>
          </div>
          <Badge variant="secondary" className="w-full justify-center">
            Contract Review
          </Badge>
        </div>
      </div>

      {/* Main Content */}
      <div className="flex-1 ml-64 p-8">
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5 }}
        >
          <div className="mb-8">
            <h1 className="text-3xl font-bold text-gray-900 mb-2">Contract Review</h1>
            <p className="text-gray-600">Analyze contracts for compliance, risks, and legal issues</p>
          </div>

          <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
            {/* Input Section */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center space-x-2">
                  <FileText className="h-5 w-5" />
                  <span>Contract Input</span>
                </CardTitle>
                <CardDescription>
                  Upload a file or paste your contract text for review
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                <div>
                  <label className="block text-sm font-medium mb-2">Upload Contract File</label>
                  <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                    <Upload className="h-8 w-8 mx-auto mb-2 text-gray-400" />
                    <p className="text-sm text-gray-600 mb-2">
                      Click to upload or drag and drop
                    </p>
                    <p className="text-xs text-gray-500">
                      Supports .txt, .doc, .docx files
                    </p>
                    <input
                      type="file"
                      accept=".txt,.doc,.docx"
                      onChange={handleFileUpload}
                      className="hidden"
                      id="file-upload"
                    />
                    <Button 
                      variant="outline" 
                      size="sm" 
                      className="mt-2"
                      onClick={() => document.getElementById('file-upload').click()}
                    >
                      Choose File
                    </Button>
                  </div>
                </div>

                <div className="text-center text-gray-500">
                  <span>or</span>
                </div>

                <div>
                  <label className="block text-sm font-medium mb-2">Paste Contract Text</label>
                  <Textarea
                    placeholder="Paste your contract text here..."
                    rows={10}
                    value={contractText}
                    onChange={(e) => setContractText(e.target.value)}
                    className="min-h-[200px]"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium mb-2">Jurisdiction</label>
                  <Select onValueChange={setJurisdiction}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select jurisdiction" />
                    </SelectTrigger>
                    <SelectContent>
                      {jurisdictions.map((jurisdiction) => (
                        <SelectItem key={jurisdiction.value} value={jurisdiction.value}>
                          {jurisdiction.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <label className="block text-sm font-medium mb-2">Review Type</label>
                  <Select onValueChange={setReviewType} defaultValue="general">
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      {reviewTypes.map((type) => (
                        <SelectItem key={type.value} value={type.value}>
                          {type.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <Button 
                  onClick={reviewContract}
                  disabled={!contractText || !jurisdiction || isReviewing}
                  className="w-full"
                >
                  {isReviewing ? (
                    <>
                      <Search className="h-4 w-4 mr-2 animate-spin" />
                      Reviewing Contract...
                    </>
                  ) : (
                    <>
                      <Search className="h-4 w-4 mr-2" />
                      Review Contract
                    </>
                  )}
                </Button>
              </CardContent>
            </Card>

            {/* Results Section */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center space-x-2">
                  <Search className="h-5 w-5" />
                  <span>Review Results</span>
                </CardTitle>
                <CardDescription>
                  Detailed analysis and recommendations
                </CardDescription>
              </CardHeader>
              <CardContent>
                {reviewResults ? (
                  <div className="space-y-6">
                    {/* Risk Level */}
                    <div className="flex items-center justify-between">
                      <span className="font-medium">Overall Risk Level:</span>
                      <Badge className={getRiskColor(reviewResults.risk_level)}>
                        {React.createElement(getRiskIcon(reviewResults.risk_level), { className: "h-4 w-4 mr-1" })}
                        {reviewResults.risk_level}
                      </Badge>
                    </div>

                    <Button onClick={downloadReport} variant="outline" size="sm" className="w-full">
                      <Download className="h-4 w-4 mr-2" />
                      Download Full Report
                    </Button>

                    {/* Summary */}
                    <div>
                      <h3 className="font-semibold mb-2">Summary</h3>
                      <p className="text-sm text-gray-600">{reviewResults.summary}</p>
                    </div>

                    {/* Compliance Issues */}
                    {reviewResults.compliance_issues.length > 0 && (
                      <Alert>
                        <AlertTriangle className="h-4 w-4" />
                        <AlertDescription>
                          <strong>Compliance Issues Found:</strong>
                          <ul className="mt-2 space-y-1">
                            {reviewResults.compliance_issues.slice(0, 2).map((issue, index) => (
                              <li key={index} className="text-sm">• {issue}</li>
                            ))}
                            {reviewResults.compliance_issues.length > 2 && (
                              <li className="text-sm text-gray-500">
                                +{reviewResults.compliance_issues.length - 2} more issues
                              </li>
                            )}
                          </ul>
                        </AlertDescription>
                      </Alert>
                    )}

                    {/* Legal Risks */}
                    {reviewResults.legal_risks.length > 0 && (
                      <div>
                        <h3 className="font-semibold mb-2 text-red-600">Legal Risks</h3>
                        <ul className="space-y-1">
                          {reviewResults.legal_risks.slice(0, 3).map((risk, index) => (
                            <li key={index} className="text-sm text-gray-600">• {risk}</li>
                          ))}
                          {reviewResults.legal_risks.length > 3 && (
                            <li className="text-sm text-gray-500">
                              +{reviewResults.legal_risks.length - 3} more risks
                            </li>
                          )}
                        </ul>
                      </div>
                    )}

                    {/* Missing Clauses */}
                    {reviewResults.missing_clauses.length > 0 && (
                      <div>
                        <h3 className="font-semibold mb-2 text-yellow-600">Missing Clauses</h3>
                        <ul className="space-y-1">
                          {reviewResults.missing_clauses.slice(0, 3).map((clause, index) => (
                            <li key={index} className="text-sm text-gray-600">• {clause}</li>
                          ))}
                          {reviewResults.missing_clauses.length > 3 && (
                            <li className="text-sm text-gray-500">
                              +{reviewResults.missing_clauses.length - 3} more clauses
                            </li>
                          )}
                        </ul>
                      </div>
                    )}

                    {/* Recommendations */}
                    {reviewResults.recommendations.length > 0 && (
                      <div>
                        <h3 className="font-semibold mb-2 text-green-600">Recommendations</h3>
                        <ul className="space-y-1">
                          {reviewResults.recommendations.slice(0, 3).map((rec, index) => (
                            <li key={index} className="text-sm text-gray-600">• {rec}</li>
                          ))}
                          {reviewResults.recommendations.length > 3 && (
                            <li className="text-sm text-gray-500">
                              +{reviewResults.recommendations.length - 3} more recommendations
                            </li>
                          )}
                        </ul>
                      </div>
                    )}
                  </div>
                ) : (
                  <div className="text-center py-12 text-gray-500">
                    <Search className="h-12 w-12 mx-auto mb-4 opacity-50" />
                    <p>Upload or paste a contract and click "Review Contract" to see analysis here</p>
                  </div>
                )}
              </CardContent>
            </Card>
          </div>
        </motion.div>
      </div>
    </div>
  );
}

export default ContractReview;

