import React, { useState, useEffect } from 'react';
import { Link, useLocation } from 'react-router-dom';
import { motion } from 'framer-motion';
import { 
  Scale, 
  FileText, 
  Search, 
  Edit3, 
  BookOpen, 
  Plus,
  TrendingUp,
  Clock,
  CheckCircle,
  AlertTriangle
} from 'lucide-react';
import { Button } from '@/components/ui/button.jsx';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card.jsx';
import { Badge } from '@/components/ui/badge.jsx';
import { API_BASE_URL } from '../config'; // Import API_BASE_URL

function Sidebar() {
  const location = useLocation();
  
  const menuItems = [
    { path: '/', icon: Scale, label: 'Dashboard' },
    { path: '/generate', icon: Plus, label: 'Generate Contract' },
    { path: '/review', icon: Search, label: 'Review Contract' },
    { path: '/clauses', icon: Edit3, label: 'Draft Clauses' },
    { path: '/templates', icon: BookOpen, label: 'Template Library' }
  ];

  return (
    <div className="w-64 bg-white shadow-lg h-screen fixed left-0 top-0 z-10">
      <div className="p-6 border-b">
        <div className="flex items-center space-x-2">
          <Scale className="h-8 w-8 text-blue-600" />
          <span className="text-2xl font-bold text-gray-900">LAWAI</span>
        </div>
      </div>
      
      <nav className="mt-6">
        {menuItems.map((item) => {
          const isActive = location.pathname === item.path;
          return (
            <Link
              key={item.path}
              to={item.path}
              className={`flex items-center space-x-3 px-6 py-3 text-gray-700 hover:bg-blue-50 hover:text-blue-600 transition-colors ${
                isActive ? 'bg-blue-50 text-blue-600 border-r-2 border-blue-600' : ''
              }`}
            >
              <item.icon className="h-5 w-5" />
              <span>{item.label}</span>
            </Link>
          );
        })}
      </nav>
    </div>
  );
}

function StatCard({ icon: Icon, title, value, description, color = "text-blue-600" }) {
  return (
    <Card>
      <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
        <CardTitle className="text-sm font-medium">{title}</CardTitle>
        <Icon className={`h-4 w-4 ${color}`} />
      </CardHeader>
      <CardContent>
        <div className="text-2xl font-bold">{value}</div>
        <p className="text-xs text-muted-foreground">{description}</p>
      </CardContent>
    </Card>
  );
}

function ContractCard({ contract }) {
  const getStatusColor = (status) => {
    switch (status) {
      case 'completed': return 'bg-green-100 text-green-800';
      case 'review': return 'bg-yellow-100 text-yellow-800';
      case 'draft': return 'bg-gray-100 text-gray-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getStatusIcon = (status) => {
    switch (status) {
      case 'completed': return CheckCircle;
      case 'review': return AlertTriangle;
      case 'draft': return Clock;
      default: return Clock;
    }
  };

  const StatusIcon = getStatusIcon(contract.status);

  return (
    <Card className="hover:shadow-md transition-shadow">
      <CardHeader>
        <div className="flex items-start justify-between">
          <div>
            <CardTitle className="text-lg">{contract.title}</CardTitle>
            <CardDescription className="mt-1">
              {contract.contract_type} • {contract.jurisdiction === 'SA' ? 'Saudi Arabia' : 'Jordan'}
            </CardDescription>
          </div>
          <Badge className={getStatusColor(contract.status)}>
            <StatusIcon className="h-3 w-3 mr-1" />
            {contract.status}
          </Badge>
        </div>
      </CardHeader>
      <CardContent>
        <div className="flex items-center text-sm text-gray-500">
          <Clock className="h-4 w-4 mr-1" />
          {new Date(contract.createdAt).toLocaleDateString()}
        </div>
      </CardContent>
    </Card>
  );
}

function Dashboard() {
  const [contracts, setContracts] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  useEffect(() => {
    const fetchContracts = async () => {
      try {
        const response = await fetch(`${API_BASE_URL}/api/contracts`);
        if (!response.ok) {
          throw new Error(`HTTP error! status: ${response.status}`);
        }
        const data = await response.json();
        setContracts(data);
      } catch (error) {
        setError(error);
        console.error("Error fetching contracts:", error);
      } finally {
        setLoading(false);
      }
    };

    fetchContracts();
  }, []);

  // Calculate stats based on fetched contracts
  const totalContracts = contracts.length;
  const pendingReviews = contracts.filter(c => c.status === 'review').length;
  const completedToday = contracts.filter(c => {
    const today = new Date();
    const contractDate = new Date(c.createdAt);
    return c.status === 'completed' && 
           contractDate.getDate() === today.getDate() &&
           contractDate.getMonth() === today.getMonth() &&
           contractDate.getFullYear() === today.getFullYear();
  }).length;
  // Assuming 'risk_level' is part of the contract object from backend for risk alerts
  const riskAlerts = contracts.filter(c => c.risk_level === 'High').length; // Adjust based on actual backend response

  if (loading) {
    return <div className="flex-1 ml-64 p-8 text-center">Loading contracts...</div>;
  }

  if (error) {
    return <div className="flex-1 ml-64 p-8 text-center text-red-500">Error: {error.message}</div>;
  }

  return (
    <div className="flex">
      <Sidebar />
      
      <div className="flex-1 ml-64 p-8">
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5 }}
        >
          {/* Header */}
          <div className="mb-8">
            <h1 className="text-3xl font-bold text-gray-900 mb-2">Dashboard</h1>
            <p className="text-gray-600">Welcome back! Here's an overview of your legal work.</p>
          </div>

          {/* Stats Grid */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
            <StatCard
              icon={FileText}
              title="Total Contracts"
              value={totalContracts}
              description="All time contracts created"
              color="text-blue-600"
            />
            <StatCard
              icon={Search}
              title="Pending Reviews"
              value={pendingReviews}
              description="Contracts awaiting review"
              color="text-yellow-600"
            />
            <StatCard
              icon={CheckCircle}
              title="Completed Today"
              value={completedToday}
              description="Contracts finalized today"
              color="text-green-600"
            />
            <StatCard
              icon={AlertTriangle}
              title="Risk Alerts"
              value={riskAlerts}
              description="High-risk contracts identified"
              color="text-red-600"
            />
          </div>

          {/* Quick Actions */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mb-8">
            <Link to="/generate">
              <Button className="w-full h-20 bg-blue-600 hover:bg-blue-700 flex flex-col items-center justify-center space-y-2">
                <Plus className="h-6 w-6" />
                <span>Generate Contract</span>
              </Button>
            </Link>
            <Link to="/review">
              <Button variant="outline" className="w-full h-20 flex flex-col items-center justify-center space-y-2">
                <Search className="h-6 w-6" />
                <span>Review Contract</span>
              </Button>
            </Link>
            <Link to="/clauses">
              <Button variant="outline" className="w-full h-20 flex flex-col items-center justify-center space-y-2">
                <Edit3 className="h-6 w-6" />
                <span>Draft Clauses</span>
              </Button>
            </Link>
            <Link to="/templates">
              <Button variant="outline" className="w-full h-20 flex flex-col items-center justify-center space-y-2">
                <BookOpen className="h-6 w-6" />
                <span>Browse Templates</span>
              </Button>
            </Link>
          </div>

          {/* Recent Contracts */}
          <div>
            <div className="flex items-center justify-between mb-6">
              <h2 className="text-2xl font-bold text-gray-900">Recent Contracts</h2>
              <Button variant="outline" size="sm">
                View All
              </Button>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              {contracts.length > 0 ? (
                contracts.map((contract) => (
                  <ContractCard key={contract.id} contract={contract} />
                ))
              ) : (
                <p className="col-span-full text-center text-gray-500">No contracts found. Generate one!</p>
              )}
            </div>
          </div>
        </motion.div>
      </div>
    </div>
  );
}

export default Dashboard;

