import React, { useState } from 'react';
import { Link } from 'react-router-dom';
import { motion } from 'framer-motion';
import { 
  Scale, 
  ArrowLeft, 
  BookOpen, 
  Search,
  Filter,
  Download,
  Eye,
  Plus
} from 'lucide-react';
import { Button } from '@/components/ui/button.jsx';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card.jsx';
import { Input } from '@/components/ui/input.jsx';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select.jsx';
import { Badge } from '@/components/ui/badge.jsx';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog.jsx';

const mockTemplates = [
  {
    id: 1,
    name: 'Employment Agreement - Saudi Arabia',
    description: 'Comprehensive employment contract template compliant with Saudi Labor Law',
    category: 'Employment',
    jurisdiction: 'SA',
    lastUpdated: '2025-06-01',
    downloads: 245,
    content: `EMPLOYMENT AGREEMENT

This Employment Agreement ("Agreement") is entered into on [DATE], between [COMPANY NAME], a company organized under the laws of Saudi Arabia ("Company"), and [EMPLOYEE NAME], an individual ("Employee").

ARTICLE 1: POSITION AND DUTIES
The Employee shall serve as [POSITION TITLE] and shall perform such duties and responsibilities as may be assigned by the Company from time to time, consistent with the Employee's position.

ARTICLE 2: TERM OF EMPLOYMENT
This Agreement shall commence on [START DATE] and shall continue until terminated in accordance with the provisions herein or as required by Saudi Labor Law.

ARTICLE 3: COMPENSATION
The Company shall pay the Employee a monthly salary of SAR [AMOUNT], payable in accordance with the Company's standard payroll practices.

ARTICLE 4: BENEFITS
The Employee shall be entitled to benefits as provided in the Company's employee handbook and as required by Saudi Labor Law, including but not limited to:
- Annual leave as per Saudi Labor Law
- Sick leave provisions
- End of service benefits
- Health insurance coverage

ARTICLE 5: CONFIDENTIALITY
The Employee acknowledges that during employment, they may have access to confidential information and agrees to maintain strict confidentiality.

ARTICLE 6: TERMINATION
This Agreement may be terminated by either party in accordance with the provisions of Saudi Labor Law.

ARTICLE 7: GOVERNING LAW
This Agreement shall be governed by and construed in accordance with the laws of Saudi Arabia.

[Additional clauses as required by Saudi Labor Law]`
  },
  {
    id: 2,
    name: 'Commercial Lease Agreement - Jordan',
    description: 'Standard commercial property lease agreement for Jordan',
    category: 'Real Estate',
    jurisdiction: 'JO',
    lastUpdated: '2025-05-28',
    downloads: 189,
    content: `COMMERCIAL LEASE AGREEMENT

This Commercial Lease Agreement ("Lease") is entered into on [DATE], between [LANDLORD NAME] ("Landlord") and [TENANT NAME] ("Tenant").

ARTICLE 1: PREMISES
Landlord hereby leases to Tenant the commercial premises located at [PROPERTY ADDRESS], Jordan ("Premises").

ARTICLE 2: TERM
The lease term shall be for [DURATION] commencing on [START DATE] and ending on [END DATE].

ARTICLE 3: RENT
Tenant shall pay monthly rent of JOD [AMOUNT], due on the [DAY] of each month.

ARTICLE 4: USE OF PREMISES
The Premises shall be used solely for [BUSINESS PURPOSE] and for no other purpose without Landlord's written consent.

ARTICLE 5: MAINTENANCE AND REPAIRS
[Maintenance responsibilities as per Jordanian law]

ARTICLE 6: GOVERNING LAW
This Lease shall be governed by the laws of Jordan.

[Additional terms and conditions]`
  },
  {
    id: 3,
    name: 'Service Agreement Template',
    description: 'General service provider agreement template',
    category: 'Service',
    jurisdiction: 'SA',
    lastUpdated: '2025-06-05',
    downloads: 156,
    content: `SERVICE AGREEMENT

This Service Agreement is entered into between [CLIENT NAME] and [SERVICE PROVIDER NAME].

[Service agreement content...]`
  },
  {
    id: 4,
    name: 'Partnership Agreement - Jordan',
    description: 'Business partnership agreement template for Jordan',
    category: 'Partnership',
    jurisdiction: 'JO',
    lastUpdated: '2025-05-30',
    downloads: 134,
    content: `PARTNERSHIP AGREEMENT

This Partnership Agreement establishes a partnership between the parties under Jordanian law.

[Partnership agreement content...]`
  },
  {
    id: 5,
    name: 'Non-Disclosure Agreement',
    description: 'Mutual non-disclosure agreement template',
    category: 'Confidentiality',
    jurisdiction: 'SA',
    lastUpdated: '2025-06-03',
    downloads: 298,
    content: `NON-DISCLOSURE AGREEMENT

This Non-Disclosure Agreement protects confidential information shared between parties.

[NDA content...]`
  },
  {
    id: 6,
    name: 'Supply Agreement - Saudi Arabia',
    description: 'Goods supply agreement template for Saudi Arabia',
    category: 'Commercial',
    jurisdiction: 'SA',
    lastUpdated: '2025-05-25',
    downloads: 167,
    content: `SUPPLY AGREEMENT

This Supply Agreement governs the supply of goods between supplier and purchaser.

[Supply agreement content...]`
  }
];

const categories = ['All', 'Employment', 'Real Estate', 'Service', 'Partnership', 'Confidentiality', 'Commercial'];
const jurisdictions = [
  { value: 'all', label: 'All Jurisdictions' },
  { value: 'SA', label: 'Saudi Arabia' },
  { value: 'JO', label: 'Jordan' }
];

function TemplateCard({ template, onView }) {
  const getJurisdictionBadge = (jurisdiction) => {
    return jurisdiction === 'SA' ? 
      <Badge variant="secondary">Saudi Arabia</Badge> : 
      <Badge variant="outline">Jordan</Badge>;
  };

  return (
    <Card className="hover:shadow-md transition-shadow">
      <CardHeader>
        <div className="flex items-start justify-between">
          <div className="flex-1">
            <CardTitle className="text-lg mb-2">{template.name}</CardTitle>
            <CardDescription className="text-sm">
              {template.description}
            </CardDescription>
          </div>
          {getJurisdictionBadge(template.jurisdiction)}
        </div>
      </CardHeader>
      <CardContent>
        <div className="flex items-center justify-between mb-4">
          <div className="flex items-center space-x-4 text-sm text-gray-500">
            <span>Category: {template.category}</span>
            <span>•</span>
            <span>{template.downloads} downloads</span>
          </div>
        </div>
        
        <div className="flex space-x-2">
          <Button 
            variant="outline" 
            size="sm" 
            onClick={() => onView(template)}
            className="flex-1"
          >
            <Eye className="h-4 w-4 mr-2" />
            View
          </Button>
          <Button 
            variant="outline" 
            size="sm"
            onClick={() => {
              const element = document.createElement('a');
              const file = new Blob([template.content], { type: 'text/plain' });
              element.href = URL.createObjectURL(file);
              element.download = `${template.name}.txt`;
              document.body.appendChild(element);
              element.click();
              document.body.removeChild(element);
            }}
          >
            <Download className="h-4 w-4" />
          </Button>
        </div>
      </CardContent>
    </Card>
  );
}

function TemplateViewer({ template, isOpen, onClose }) {
  if (!template) return null;

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-4xl max-h-[80vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle>{template.name}</DialogTitle>
          <DialogDescription>
            {template.description}
          </DialogDescription>
        </DialogHeader>
        
        <div className="space-y-4">
          <div className="flex items-center space-x-4">
            <Badge variant="secondary">{template.category}</Badge>
            <Badge variant={template.jurisdiction === 'SA' ? 'secondary' : 'outline'}>
              {template.jurisdiction === 'SA' ? 'Saudi Arabia' : 'Jordan'}
            </Badge>
            <span className="text-sm text-gray-500">
              Last updated: {new Date(template.lastUpdated).toLocaleDateString()}
            </span>
          </div>
          
          <div className="bg-gray-50 p-4 rounded-lg max-h-96 overflow-y-auto">
            <pre className="whitespace-pre-wrap text-sm text-gray-800">
              {template.content}
            </pre>
          </div>
          
          <div className="flex space-x-2">
            <Button 
              onClick={() => {
                const element = document.createElement('a');
                const file = new Blob([template.content], { type: 'text/plain' });
                element.href = URL.createObjectURL(file);
                element.download = `${template.name}.txt`;
                document.body.appendChild(element);
                element.click();
                document.body.removeChild(element);
              }}
            >
              <Download className="h-4 w-4 mr-2" />
              Download Template
            </Button>
            <Button variant="outline" onClick={onClose}>
              Close
            </Button>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}

function TemplateLibrary() {
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedCategory, setSelectedCategory] = useState('All');
  const [selectedJurisdiction, setSelectedJurisdiction] = useState('all');
  const [selectedTemplate, setSelectedTemplate] = useState(null);
  const [isViewerOpen, setIsViewerOpen] = useState(false);

  const filteredTemplates = mockTemplates.filter(template => {
    const matchesSearch = template.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         template.description.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesCategory = selectedCategory === 'All' || template.category === selectedCategory;
    const matchesJurisdiction = selectedJurisdiction === 'all' || template.jurisdiction === selectedJurisdiction;
    
    return matchesSearch && matchesCategory && matchesJurisdiction;
  });

  const handleViewTemplate = (template) => {
    setSelectedTemplate(template);
    setIsViewerOpen(true);
  };

  return (
    <div className="flex">
      {/* Sidebar */}
      <div className="w-64 bg-white shadow-lg h-screen fixed left-0 top-0 z-10">
        <div className="p-6 border-b">
          <Link to="/" className="flex items-center space-x-2 text-gray-600 hover:text-gray-900">
            <ArrowLeft className="h-5 w-5" />
            <span>Back to Dashboard</span>
          </Link>
        </div>
        <div className="p-6">
          <div className="flex items-center space-x-2 mb-4">
            <Scale className="h-6 w-6 text-blue-600" />
            <span className="text-xl font-bold text-gray-900">LAWAI</span>
          </div>
          <Badge variant="secondary" className="w-full justify-center">
            Template Library
          </Badge>
        </div>
      </div>

      {/* Main Content */}
      <div className="flex-1 ml-64 p-8">
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5 }}
        >
          <div className="mb-8">
            <h1 className="text-3xl font-bold text-gray-900 mb-2">Template Library</h1>
            <p className="text-gray-600">Browse and download legal contract templates</p>
          </div>

          {/* Search and Filters */}
          <Card className="mb-8">
            <CardHeader>
              <CardTitle className="flex items-center space-x-2">
                <Search className="h-5 w-5" />
                <span>Search Templates</span>
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div>
                  <Input
                    placeholder="Search templates..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="w-full"
                  />
                </div>
                
                <div>
                  <Select onValueChange={setSelectedCategory} defaultValue="All">
                    <SelectTrigger>
                      <SelectValue placeholder="Category" />
                    </SelectTrigger>
                    <SelectContent>
                      {categories.map((category) => (
                        <SelectItem key={category} value={category}>
                          {category}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                
                <div>
                  <Select onValueChange={setSelectedJurisdiction} defaultValue="all">
                    <SelectTrigger>
                      <SelectValue placeholder="Jurisdiction" />
                    </SelectTrigger>
                    <SelectContent>
                      {jurisdictions.map((jurisdiction) => (
                        <SelectItem key={jurisdiction.value} value={jurisdiction.value}>
                          {jurisdiction.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Results Summary */}
          <div className="flex items-center justify-between mb-6">
            <p className="text-gray-600">
              Showing {filteredTemplates.length} template{filteredTemplates.length !== 1 ? 's' : ''}
            </p>
            <Link to="/generate">
              <Button>
                <Plus className="h-4 w-4 mr-2" />
                Create New Contract
              </Button>
            </Link>
          </div>

          {/* Templates Grid */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {filteredTemplates.map((template) => (
              <TemplateCard 
                key={template.id} 
                template={template} 
                onView={handleViewTemplate}
              />
            ))}
          </div>

          {filteredTemplates.length === 0 && (
            <div className="text-center py-12">
              <BookOpen className="h-12 w-12 mx-auto mb-4 text-gray-400" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">No templates found</h3>
              <p className="text-gray-600">
                Try adjusting your search criteria or browse all templates
              </p>
            </div>
          )}

          {/* Template Viewer Dialog */}
          <TemplateViewer 
            template={selectedTemplate}
            isOpen={isViewerOpen}
            onClose={() => setIsViewerOpen(false)}
          />
        </motion.div>
      </div>
    </div>
  );
}

export default TemplateLibrary;

