from flask_sqlalchemy import SQLAlchemy
from datetime import datetime
from src.models.user import db

class Contract(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    title = db.Column(db.String(200), nullable=False)
    content = db.Column(db.Text, nullable=False)
    contract_type = db.Column(db.String(100), nullable=False)
    jurisdiction = db.Column(db.String(50), nullable=False)  # SA, JO, or general
    status = db.Column(db.String(50), default='draft')  # draft, reviewed, finalized
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    updated_at = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    user_id = db.Column(db.Integer, db.ForeignKey('user.id'), nullable=False)
    
    # Relationship with User
    user = db.relationship('User', backref=db.backref('contracts', lazy=True))

    def __repr__(self):
        return f'<Contract {self.title}>'

    def to_dict(self):
        return {
            'id': self.id,
            'title': self.title,
            'content': self.content,
            'contract_type': self.contract_type,
            'jurisdiction': self.jurisdiction,
            'status': self.status,
            'created_at': self.created_at.isoformat() if self.created_at else None,
            'updated_at': self.updated_at.isoformat() if self.updated_at else None,
            'user_id': self.user_id
        }

class Template(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(200), nullable=False)
    description = db.Column(db.Text)
    content = db.Column(db.Text, nullable=False)
    category = db.Column(db.String(100), nullable=False)
    jurisdiction = db.Column(db.String(50), nullable=False)  # SA, JO, or general
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    
    def __repr__(self):
        return f'<Template {self.name}>'

    def to_dict(self):
        return {
            'id': self.id,
            'name': self.name,
            'description': self.description,
            'content': self.content,
            'category': self.category,
            'jurisdiction': self.jurisdiction,
            'created_at': self.created_at.isoformat() if self.created_at else None
        }

class ReviewResult(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    contract_id = db.Column(db.Integer, db.ForeignKey('contract.id'), nullable=False)
    review_type = db.Column(db.String(50), nullable=False)  # compliance, risk, general
    findings = db.Column(db.Text, nullable=False)  # JSON string of findings
    recommendations = db.Column(db.Text)  # JSON string of recommendations
    risk_level = db.Column(db.String(20))  # low, medium, high
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    
    # Relationship with Contract
    contract = db.relationship('Contract', backref=db.backref('reviews', lazy=True))

    def __repr__(self):
        return f'<ReviewResult for Contract {self.contract_id}>'

    def to_dict(self):
        return {
            'id': self.id,
            'contract_id': self.contract_id,
            'review_type': self.review_type,
            'findings': self.findings,
            'recommendations': self.recommendations,
            'risk_level': self.risk_level,
            'created_at': self.created_at.isoformat() if self.created_at else None
        }


